<?php

if (!defined('ABSPATH')) {
	exit;
}

class HandleStepTwo
{

	private $zip_file;

	public function __construct()
	{

		add_action('wp_ajax_wp_pack_step2', [$this, 'handle_step']);
	}

	public function handle_step()
	{

		// Checked POST nonce is not empty.
		if (empty($_POST['nonce'])) wp_die('0');

		$nonce = sanitize_key(wp_unslash($_POST['nonce']));

        if (!wp_verify_nonce($nonce, 'install_wp_package')) {

			echo wp_json_encode(new WP_Error('error_data', 'Invalid nonce', array('status' => 403))); // {"errors":{"invalid_nonce":["Invalid nonce"]},"error_data":{"invalid_nonce":{"status":403}}}

			wp_die();
		}

		/**
		 * Execution code here
		 */
		if(empty($_POST['packages'])) {

            echo wp_json_encode(new WP_Error('error_data', 'Missing of the packages', array('status' => 404)));

            wp_die();
        }
		$packages   = wpPackagesSanitizePostArray($_POST['packages']);

		if(empty($_POST['packageId'])) {

            echo wp_json_encode(new WP_Error('error_data', 'Missing of the packageId', array('status' => 404)));

            wp_die();
        }
		$package_id = sanitize_text_field(wp_unslash($_POST['packageId']));

		// Disable Cron
		(new WP_CRON_Manager)->disable_wp_cron();

		// Download package
		$this->downloadPackage($packages[$package_id]);

		// Unzip package
		$this->unzipPackage();

		// Success
		echo wp_json_encode(new WP_REST_Response([
			'success' => true,
			'message' => 'Downloading package...'//'Package downloaded successfully'
		], 200)); // {"data":{"success":true,"message":"This is a success"},"headers":[],"status":200}

		wp_die();
	}

	public function downloadPackage($package)
	{

		$url = $package['download_url'];
		$upload_dir = wp_upload_dir();
		$file_name = basename(wp_parse_url($url, PHP_URL_PATH));

		$this->zip_file = $upload_dir['path'] . '/' . $file_name;

		// Download the file using WordPress HTTP API
		$response = wp_remote_get($url, array(
			'timeout' => 300,
			'stream' => true,
			'filename' => $this->zip_file
		));

		if (is_wp_error($response)) {

			wp_delete_file($this->zip_file);

			echo wp_json_encode(new WP_Error('error_data', $response->get_error_message(), array('status' => 500)));

			wp_die();
		}

		$response_code = wp_remote_retrieve_response_code($response);

		if (200 != $response_code) {

			wp_delete_file($this->zip_file);

			echo wp_json_encode(new WP_Error('error_data', 'Failed to download package. HTTP response code: ' . $response_code, array('status' => 500)));

			wp_die();
		}
	}

	public function unzipPackage()
	{

		$upload_dir = wp_upload_dir();

		$random_suffix = sprintf('%05d', wp_rand(0, 99999));
		$extract_path = $upload_dir['path'] . '/extracted-package-' . $random_suffix;

		if (!class_exists('ZipArchive')) {

			wp_delete_file($this->zip_file);

			echo wp_json_encode(new WP_Error('error_data', 'ZipArchive is not available on your system. Please contact your hosting provider.', array('status' => 404)));

			wp_die();
		}

		$zip = new ZipArchive;
		$res = $zip->open($this->zip_file);

		if ($res !== TRUE) {

			wp_delete_file($this->zip_file);

			echo wp_json_encode(new WP_Error('error_data', 'Failed to open the zip file. Error code: ' . $res, array('status' => 500)));

			wp_die();
		}

		// Create the extraction directory if it doesn't exist
		wp_mkdir_p($extract_path);

		// Extract the contents
		$extraction_result = $zip->extractTo($extract_path);
		$zip->close();

		if ($extraction_result === false) {

			wp_delete_file($this->zip_file);

			echo wp_json_encode(new WP_Error('error_data', 'Failed to extract the zip file.', array('status' => 500)));

			wp_die();
		}

		// Delete the zip file after extraction
		wp_delete_file($this->zip_file);

		// Save the extract path for future stages to use
		update_option('wp_package_extract_path', $extract_path);
	}
}
