<?php

if (!defined('ABSPATH')) {
    exit;
}

class HandleStepSix
{

    public function __construct()
    {

        add_action('wp_ajax_wp_pack_step6', [$this, 'handle_step']);
    }

    public function handle_step()
    {

        // Checked POST nonce is not empty.
        if (empty($_POST['nonce'])) wp_die('0');

        $nonce = sanitize_key(wp_unslash($_POST['nonce']));

        if (!wp_verify_nonce($nonce, 'install_wp_package')) {

            echo wp_json_encode(new WP_Error('error_data', 'Invalid nonce', array('status' => 403))); // {"errors":{"invalid_nonce":["Invalid nonce"]},"error_data":{"invalid_nonce":{"status":403}}}

            wp_die();
        }

        /**
         * Execution code here
         */
        // Disable Cron
        (new WP_CRON_Manager)->disable_wp_cron();

        $this->deletePostsData();

        $this->importContent();

        // Success
        echo wp_json_encode(new WP_REST_Response([
            'success' => true,
            'message' => 'Importing the content (can take a few minutes)...' //'Content Imported successfully'
        ], 200)); // {"data":{"success":true,"message":"This is a success"},"headers":[],"status":200}

        wp_die();
    }

    public function importContent()
    {

        $extract_path = get_option('wp_package_extract_path');

        if (!$extract_path) {

            echo wp_json_encode(new WP_Error('error_data', 'Not Path Option', array('status' => 404)));

            wp_die();
        }

        // Download and activate WordPress Importer plugin
        include_once(ABSPATH . 'wp-admin/includes/plugin-install.php');
        include_once(ABSPATH . 'wp-admin/includes/class-wp-upgrader.php');
        include_once(ABSPATH . 'wp-admin/includes/misc.php');
        include_once(ABSPATH . 'wp-admin/includes/file.php');

        $plugin_slug = 'wordpress-importer';
        $plugin = $plugin_slug . '/wordpress-importer.php';

        if (!is_plugin_active($plugin)) {

            $api = plugins_api('plugin_information', array('slug' => $plugin_slug));

            if (is_wp_error($api)) {
                
                // Check if plugin files already exist (might be installed but not active)
                $plugin_file = WP_PLUGIN_DIR . '/' . $plugin;
                if (file_exists($plugin_file)) {
                    // Plugin exists, just activate it
                    $activation_result = activate_plugin($plugin, '', false, true);
                    if (is_wp_error($activation_result)) {
                        echo wp_json_encode(new WP_Error('error_data', 'Failed to activate existing importer plugin: ' . $activation_result->get_error_message(), array('status' => 500)));
                        wp_die();
                    }
                } else {
                    // Plugin doesn't exist and we can't download it
                    echo wp_json_encode(new WP_Error('error_data', 'WordPress Importer plugin is required but could not be downloaded due to connection issues. Please install it manually from the WordPress admin or check your server\'s connection to WordPress.org. Error: ' . $api->get_error_message(), array('status' => 500)));
                    wp_die();
                }
            } else {
                // API call succeeded, proceed with installation
                $upgrader = new Plugin_Upgrader(new Quiet_Skin());
                $result = $upgrader->install($api->download_link);

                if (is_wp_error($result)) {

                    echo wp_json_encode(new WP_Error('error_data', 'Failed to install importer plugin: ' . $result->get_error_message(), array('status' => 500)));

                    wp_die();
                }

                activate_plugin($plugin, '', false, true);
            }
        }

        // Ensure WP_LOAD_IMPORTERS is true so that importer can run
        if (!defined('WP_LOAD_IMPORTERS')) {

            define('WP_LOAD_IMPORTERS', true);
        }

        // Load WP Importer plugin & include all necessary parser classes & the WP_Import() class
        if (file_exists(ABSPATH . 'wp-admin/includes/class-wp-importer.php')) {

            require_once ABSPATH . 'wp-admin/includes/class-wp-importer.php';
        }

        $importer_path = WP_PLUGIN_DIR . '/wordpress-importer';
        
        // Load the PHP toolkit first (required for WPURL class)
        $php_toolkit_loader = $importer_path . '/php-toolkit/load.php';
        if (file_exists($php_toolkit_loader)) {
            require_once($php_toolkit_loader);
        } else {
            echo wp_json_encode(new WP_Error('error_data', 'PHP toolkit loader not found at ' . $php_toolkit_loader, array('status' => 404)));
            wp_die();
        }
        
        $needed_files = [
            $importer_path . '/wordpress-importer.php',
            $importer_path . '/parsers/class-wxr-parser.php',
            $importer_path . '/parsers/class-wxr-parser-simplexml.php',
            $importer_path . '/parsers/class-wxr-parser-xml.php',
            $importer_path . '/parsers/class-wxr-parser-regex.php',
            $importer_path . '/class-wp-import.php',
        ];

        foreach ($needed_files as $file) {

            if (file_exists($file)) {

                require_once($file);
            } else {

                echo wp_json_encode(new WP_Error('error_data', 'Required dependency file not found at ' . $file, array('status' => 404)));

                wp_die();
            }
        }

        // Find the XML file
        $xml_files = glob($extract_path . '/*.xml');

        if (empty($xml_files)) {

            echo wp_json_encode(new WP_Error('error_data', 'No XML file for importing content was found in the extracted package.', array('status' => 404)));

            wp_die();
        }
        $xml_file = $xml_files[0]; // Get the first (and only) XML file found

        // Import the content (first check if the importer class is available)
        if (class_exists('WP_Import')) {

            /** @var WP_Import $importer */
            $importer = new WP_Import();
            $importer->fetch_attachments = true;
            $importer->import($xml_file);
        } else {

            echo wp_json_encode(new WP_Error('error_data', 'Failed to load the WP_Import class.', array('status' => 500)));

            wp_die();
        }
    }

    public function deletePostsData()
    {

        // Delete posts
        wpPackagesCleanPosts();

        // Set permalinks
        wpPackagesSetPermalinkStructure('/%postname%/');

        // Delete all categories and tags
        wpPackagesCleanTerms();

        // Delete all comments
        wpPackagesCleanComments();
    }
}
