<?php

if (!defined('ABSPATH')) {
    exit;
}

class HandleStepSeven
{

    public function __construct()
    {

        add_action('wp_ajax_wp_pack_step7', [$this, 'handle_step']);
    }

    public function handle_step()
    {

        // Checked POST nonce is not empty.
        if (empty($_POST['nonce'])) wp_die('0');

        $nonce = sanitize_key(wp_unslash($_POST['nonce']));

        if (!wp_verify_nonce($nonce, 'install_wp_package')) {

            echo wp_json_encode(new WP_Error('error_data', 'Invalid nonce', array('status' => 403))); // {"errors":{"invalid_nonce":["Invalid nonce"]},"error_data":{"invalid_nonce":{"status":403}}}

            wp_die();
        }

        /**
         * Execution code here
         */
        // Disable Cron
        (new WP_CRON_Manager)->disable_wp_cron();

        if(empty($_POST['packages'])) {

            echo wp_json_encode(new WP_Error('error_data', 'Missing of the packages', array('status' => 404)));

            wp_die();
        }
        $packages   = wpPackagesSanitizePostArray($_POST['packages']);

        if(empty($_POST['packageId'])) {

            echo wp_json_encode(new WP_Error('error_data', 'Missing of the packageId', array('status' => 404)));

            wp_die();
        }
        
        $package_id = sanitize_text_field(wp_unslash($_POST['packageId']));

        // update wpforms fields
        $this->update_wpform_from_fields($packages[$package_id]);

        // Dismiss ads for wpforms
        $this->dismiss_wpforms_ads();

        // Clean up
        $this->cleanAndCustomize($packages[$package_id]);

        // Success
        echo wp_json_encode(new WP_REST_Response([
            'success' => true,
            'message' => 'Cleaning up post-installation...'
        ], 200)); // {"data":{"success":true,"message":"This is a success"},"headers":[],"status":200}

        wp_die();
    }

    public function cleanAndCustomize($package)
    {

        $extract_path = get_option('wp_package_extract_path');

        if (!$extract_path) {

            echo wp_json_encode(new WP_Error('error_data', 'Not Path Option', array('status' => 404)));

            wp_die();
        }

        // Deactivate & delete importer plugin
        $importer_plugin = 'wordpress-importer/wordpress-importer.php';

        if (is_plugin_active($importer_plugin)) {

            deactivate_plugins($importer_plugin);
        }

        if (file_exists(WP_PLUGIN_DIR . '/' . $importer_plugin)) {

            delete_plugins(array($importer_plugin));
        }

        // Update the site's name to the package's name
        if (get_option('blogname') !== false) {

            update_option('blogname', $package['name']);
        }

        // Set the homepage as static Home if it is specified to do so in package_config
        if (isset($package['homepage_title'])) {

            $homepage_title = $package['homepage_title'];

            $query = new WP_Query([
                'post_type' => 'page',
                'title'     => $homepage_title,
                'posts_per_page' => 1
            ]);

            if ($query->have_posts()) {

                $query->the_post();
                $home_page_id = get_the_ID();
                wp_reset_postdata();

                if (get_option('show_on_front') !== false) {

                    update_option('show_on_front', 'page');
                }

                if (get_option('page_on_front') !== false) {

                    update_option('page_on_front', $home_page_id);
                }
            }
        }

        // Find and replace links to change old site URL to new site URL - including for image URLs
        $this->find_and_replace_links($extract_path);

        // Hide screen options for the current user
        $current_user_id = get_current_user_id();

        if ($current_user_id) {

            update_user_meta($current_user_id, 'show_welcome_panel', 0);

            $hidden_metaboxes = [
                'dashboard_site_health',                                                                                // Site Health Status
                'dashboard_right_now',                                                                                    // At a Glance
                'dashboard_activity',                                                                                    // Activity
                'dashboard_quick_press',                                                                                // Quick Draft
                'dashboard_primary',                                                                                    // WordPress Events and News
                'wpforms_reports_widget_lite',                                                                            // WPForms Lite reports widget
                //	'woocommerce_dashboard_status',																			// WooCommerce status
                //	'woocommerce_dashboard_recent_reviews',																	// WooCommerce recent reviews
                //	'e-dashboard-overview'																					// Elementor overview
            ];

            update_user_meta($current_user_id, 'metaboxhidden_dashboard', $hidden_metaboxes);
        }
    }

    public function find_and_replace_links($extract_path)
    {

        // Find source site's URL from XML import file and if we can't, then skip this whole find and replace step
        $old_site_url = $this->get_base_site_url_from_xml($extract_path);

        if ($old_site_url === null) {

            error_log('WP Package Importer: Skipping find and replace, could not determine old site URL.');
            return;
        }

        // If we found it, continue by setting up the new site's URL and replacement setup
        $new_site_url = home_url();
        $new_site_path = wp_parse_url($new_site_url, PHP_URL_PATH) ?: '';
        $wp_in_subfolder = !empty($new_site_path);

        // Always update hardcoded links inside content (link hrefs and image srcs)
        $this->handle_hardcoded_links_in_content($old_site_url, $new_site_url);

        // If WP is installed in a subfolder, also update relative links in content and navigation
        if ($wp_in_subfolder) {

            $this->handle_relative_links_in_content($new_site_path);
            $this->handle_relative_links_in_navigation();
        }
        wp_cache_flush();
    }

    public function get_base_site_url_from_xml($extract_path)
    {

        $xml_files = glob($extract_path . '/*.xml');
        if (empty($xml_files)) {

            error_log('WP Package Importer: No XML file for importing content was found in the extracted package.');
            return null;
        }
        $xml_file = $xml_files[0];

        if (!file_exists($xml_file)) {

            error_log('WP Package Importer: XML file does not exist.');
            return null;
        }

        $xml = simplexml_load_file($xml_file);

        if ($xml === false) {

            error_log('WP Package Importer: Failed to parse XML file.');
            return null;
        }

        $namespaces = $xml->getNamespaces(true);

        if (!isset($namespaces['wp'])) {

            error_log('WP Package Importer: WordPress namespace not found in XML file.');
            return null;
        }

        $wp = $xml->channel->children($namespaces['wp']);
        $base_site_url = (string)$wp->base_site_url;

        if (!$base_site_url) {

            error_log('WP Package Importer: failed to get old site URL from XML (wp:base_site_url not found).');
            return null;
        }

        return $base_site_url;
    }

    public function handle_hardcoded_links_in_content($old_site_url, $new_site_url)
    {

        global $wpdb;

        $tables_and_columns = [
            $wpdb->prefix . 'options' => ['option_value'],
            $wpdb->prefix . 'posts' => ['guid', 'post_content'],
            $wpdb->prefix . 'postmeta' => ['meta_value'],
            $wpdb->prefix . 'comments' => ['comment_content', 'comment_author_url'],
            $wpdb->prefix . 'links' => ['link_url', 'link_image']
        ];

        // Replace all hardcoded occurrences of the old site's URL, with the new site's URL
        foreach ($tables_and_columns as $table => $columns) {

            foreach ($columns as $column) {

                $wpdb->query(
                    $wpdb->prepare(
                        "UPDATE {$table} SET {$column} = REPLACE({$column}, %s, %s)",
                        $old_site_url,
                        $new_site_url
                    )
                );
            }
        }
    }

    public function handle_relative_links_in_content($new_site_path)
    {

        global $wpdb;

        $tables_and_columns = [
            $wpdb->prefix . 'options' => ['option_value'],
            $wpdb->prefix . 'posts' => ['guid', 'post_content'],
            $wpdb->prefix . 'postmeta' => ['meta_value'],
            $wpdb->prefix . 'comments' => ['comment_content', 'comment_author_url'],
            $wpdb->prefix . 'links' => ['link_url', 'link_image']
        ];

        // Insert WP subfolder in relative path, replacing 'href="/' with 'href="[new_site_path]/' - excludes columns that have absolute URLs (NOT LIKE '%href="http%' or 'https%')
        foreach ($tables_and_columns as $table => $columns) {

            foreach ($columns as $column) {

                $table = sanitize_text_field($table);
                $column = sanitize_text_field($column);

                $wpdb->query($wpdb->prepare(
                    "UPDATE {$table} SET {$column} = REPLACE({$column}, %s, %s) WHERE {$column} LIKE %s AND {$column} NOT LIKE %s AND {$column} NOT LIKE %s",
                    'href="/',
                    'href="' . $new_site_path . '/',
                    '%href="/%',
                    '%href="http%',
                    '%href="https%'
                ));

                $wpdb->query($wpdb->prepare(
                    "UPDATE {$table} SET {$column} = REPLACE({$column}, %s, %s) WHERE {$column} LIKE %s AND {$column} NOT LIKE %s AND {$column} NOT LIKE %s",
                    'src="/',
                    'src="' . $new_site_path . '/',
                    '%src="/%',
                    '%src="http%',
                    '%src="https%'
                ));
            }
        }
    }

    public function handle_relative_links_in_navigation()
    {

        $navigation_posts = get_posts(array(
            'post_type' => 'wp_navigation',
            'numberposts' => -1,
        ));

        foreach ($navigation_posts as $nav_post) {

            $content = $nav_post->post_content;
            $blocks = parse_blocks($content);
            $updated = false;

            self::update_links_in_blocks($blocks, $updated);

            if ($updated) {

                $updated_content = serialize_blocks($blocks);

                wp_update_post(array(
                    'ID' => $nav_post->ID,
                    'post_content' => $updated_content,
                ));
            }
        }
    }

    public function update_links_in_blocks(&$blocks, &$updated)
    {

        foreach ($blocks as &$block) {

            if ($block['blockName'] === 'core/navigation-link' || $block['blockName'] === 'core/navigation-submenu') {

                if (isset($block['attrs']['url'])) {

                    $block['attrs']['url'] = self::update_url($block['attrs']['url']);
                    $updated = true;
                }
            }

            // Recursively update links inside inner blocks
            if (!empty($block['innerBlocks'])) {

                $this->update_links_in_blocks($block['innerBlocks'], $updated);
            }
        }
    }

    public function update_url($url)
    {

        $parsed_url = wp_parse_url($url);

        if (isset($parsed_url['path']) && substr($parsed_url['path'], 0, 1) === '/') {

            $home_url_path = wp_parse_url(home_url(), PHP_URL_PATH);
            $home_url_path = $home_url_path !== null ? $home_url_path : '';
            return (rtrim($home_url_path, '/') . $parsed_url['path']);
        }

        return $url;
    }

    private function update_wpform_from_fields($package)
    {

        if (!$package || !isset($package['name'])) {

            return;
        }

        $site_url = get_site_url();
        $domain = wp_parse_url($site_url, PHP_URL_HOST);
        $forms = wpforms()->form->get();

        foreach ($forms as $form) {

            $updated = false;
            $form_data = wpforms_decode($form->post_content);

            if (isset($form_data['settings']['notifications']) && is_array($form_data['settings']['notifications'])) {

                // &$notification allows us to make direct changes to $form_data without losing them
                foreach ($form_data['settings']['notifications'] as &$notification) {

                    $notification['sender_name'] = $package['name'];
                    $notification['sender_address'] = "no-reply@$domain";
                    $updated = true;
                }

                if ($updated) {

                    wpforms()->form->update($form->ID, $form_data);
                }
            }
        }
    }

    private function dismiss_wpforms_ads() {
		// Configure WPForms settings (none of these keys exist by default so we can't check for them first, but it won't stop and error out)
		$wpforms_settings = get_option('wpforms_settings', array());
		$wpforms_settings['hide-announcements'] = '1';
		$wpforms_settings['hide-admin-bar'] = '1';
		$wpforms_settings['email-summaries-disable'] = '1';
		$wpforms_settings['uninstall-data'] = '1';
		$wpforms_settings['modern-markup'] = '1';					// Add modern styling
		$wpforms_settings['modern-markup-is-set'] = '1';			// Optional, this should already be 1 by default
		update_option('wpforms_settings', $wpforms_settings);

		// Mark the "Getting Started" step as completed (avoid annoying redirect)
		update_option('wpforms_activation_redirect', true);

		// Dismiss upgrade to pro top bar ad & dismiss "entries are not stored in Lite" upsell distraction & in form editor lite connect popup
		$all_users = get_users(['fields' => 'ID']);
		foreach ($all_users as $user_id) {
			$dismissed = get_user_meta($user_id, 'wpforms_dismissed', true);
			if (!is_array($dismissed)) {
				$dismissed = array();
			}
			$dismissed['edu-admin-notice-bar'] = true;
			$dismissed['edu-admin-did-you-know-overview'] = true;
			$dismissed['edu-builder-lite-connect-top-bar'] = true;
			update_user_meta($user_id, 'wpforms_dismissed', $dismissed);
		}

		// Dismiss pointer pop-up ads (check existence first)
		$pointers = get_option('wpforms_pointers', []);
		$pointers['dismiss'] = ['wpforms_education_pointers_payments'];
		update_option('wpforms_pointers', $pointers);
	}
}
