<?php

if (!defined('ABSPATH')) {
    exit;
}

class HandleStepEight
{

    public function __construct()
    {

        add_action('wp_ajax_wp_pack_step8', [$this, 'handle_step']);
    }

    public function handle_step()
    {

        // Checked POST nonce is not empty.
        if (empty($_POST['nonce'])) wp_die('0');

        $nonce = sanitize_key(wp_unslash($_POST['nonce']));

        if (!wp_verify_nonce($nonce, 'install_wp_package')) {

            echo wp_json_encode(new WP_Error('error_data', 'Invalid nonce', array('status' => 403))); // {"errors":{"invalid_nonce":["Invalid nonce"]},"error_data":{"invalid_nonce":{"status":403}}}

            wp_die();
        }

        /**
         * Execution code here
         */
        if(empty($_POST['packages'])) {

            echo wp_json_encode(new WP_Error('error_data', 'Missing of the packages', array('status' => 404)));

            wp_die();
        }
        $packages   = wpPackagesSanitizePostArray($_POST['packages']);

        if(empty($_POST['packageId'])) {

            echo wp_json_encode(new WP_Error('error_data', 'Missing of the packageId', array('status' => 404)));

            wp_die();
        }
        $package_id = sanitize_text_field(wp_unslash($_POST['packageId']));

        $package = $packages[$package_id];

        if (isset($package['post_import_steps']) && is_array($package['post_import_steps']) && !empty($package['post_import_steps'])) {

            $this->handle_post_import_steps($package);
        }

        // Enable Cron
        (new WP_CRON_Manager)->re_enable_wp_cron();

        // Set permalinks
        wpPackagesSetPermalinkStructure('/%postname%/');

        // Delete extract_path file
        $this->delete_extract_path_file();

        // Success
        echo wp_json_encode(new WP_REST_Response([
            'success' => true,
            'message' => 'Cleaning up post-installation...' //All done! Your new website is ready.
        ], 200)); // {"data":{"success":true,"message":"This is a success"},"headers":[],"status":200}

        wp_die();
    }

    public function delete_extract_path_file()
    {

        $extract_path = get_option('wp_package_extract_path');

        if (is_dir($extract_path)) {

            try {

                $del = wpPackDeleteDirectory($extract_path);

                // wpPackagesDebug('$del ex p: ', $del);
            } catch (Exception $e) {

                // wpPackagesDebug("Delete extract path Error: " . $e->getMessage());
            }
        }
    }

    public function handle_post_import_steps($package)
    {

        $extract_path = get_option('wp_package_extract_path');

        if (!$extract_path) {

            echo wp_json_encode(new WP_Error('error_data', 'Not Path Option', array('status' => 404)));

            wp_die();
        }

        $steps = $package['post_import_steps'];

        foreach ($steps as $step) {

            switch ($step) {
                case 'setup_and_import_wpforms':
                    $this->setup_and_import_wpforms($package, $extract_path);
                    break;
                case 'update_query_loop_category_ids':
                    $this->update_query_loop_category_ids();
                    break;
            }
        }
    }

    public function setup_and_import_wpforms($package, $extract_path)
    {

        // Check that WPForms is active
        if (!is_plugin_active('wpforms-lite/wpforms.php')) {

            echo wp_json_encode(new WP_Error('error_data', 'WPForms plugin is not active.', array('status' => 403)));

            wp_die();
        }

        // Ensure WPForms is loaded and attempt to fully load it if not
        if (!did_action('wpforms_loaded')) {

            do_action('plugins_loaded');
        }

        require_once WPFORMS_PLUGIN_DIR . 'includes/class-form.php';

        // Delete existing WPForms
        $forms = wpforms()->form->get();

        if (!empty($forms)) {

            foreach ($forms as $form) {

                wpforms()->form->delete($form->ID);
            }
        }

        // Import new WPForms from package
        $wpforms_folder = $extract_path . '/wpforms';
        $json_files = glob($wpforms_folder . '/*.json');

        if (empty($json_files)) {

            echo wp_json_encode(new WP_Error('error_data', 'No WPForms JSON files found for import.', array('status' => 403)));

            wp_die();
        }

        // Loop through all existing json import files inside the wpforms folder
        foreach ($json_files as $json_file) {

            if (class_exists('WPForms\Admin\Tools\Views\Import')) {

                $importer = new \WPForms\Admin\Tools\Views\Import();
                $result = $importer->import_forms($json_file);

                if (is_wp_error($result)) {

                    echo wp_json_encode(new WP_Error('error_data', "Error importing forms: " . $result->get_error_message(), array('status' => 500)));

                    wp_die();
                }
            } else {

                echo wp_json_encode(new WP_Error('error_data', 'WPForms Import class not found.', array('status' => 500)));

                wp_die();
            }
        }

        // Configure WPForms plugin general settings & hide all distractions out of the gate
        $wpforms_settings = get_option('wpforms_settings', array());
        $wpforms_settings['hide-announcements'] = '1';
        $wpforms_settings['hide-admin-bar'] = '1';
        $wpforms_settings['email-summaries-disable'] = '1';
        $wpforms_settings['uninstall-data'] = '1';
        $wpforms_settings['modern-markup'] = '1';                    // Add modern styling
        $wpforms_settings['modern-markup-is-set'] = '1';            // Optional, this should already be 1 by default
        update_option('wpforms_settings', $wpforms_settings);

        // Mark the "Getting Started" step as completed (avoid annoying redirect)
        update_option('wpforms_activation_redirect', true);

        // Dismiss upgrade to pro top bar ad & dismiss "entries are not stored in Lite" upsell distraction & in form editor lite connect popup
        $all_users = get_users(['fields' => 'ID']);

        foreach ($all_users as $user_id) {

            $dismissed = get_user_meta($user_id, 'wpforms_dismissed', true);

            if (!is_array($dismissed)) {

                $dismissed = array();
            }

            $dismissed['edu-admin-notice-bar'] = true;
            $dismissed['edu-admin-did-you-know-overview'] = true;
            $dismissed['edu-builder-lite-connect-top-bar'] = true;

            update_user_meta($user_id, 'wpforms_dismissed', $dismissed);
        }

        // Dismiss pointer pop-up ads (check existence first)
        $pointers = get_option('wpforms_pointers', []);
        $pointers['dismiss'] = ['wpforms_education_pointers_payments'];

        update_option('wpforms_pointers', $pointers);

        // Update form IDs inside the content (pages and templates)
        if (!$package || !isset($package['update_wpform_ids'])) {

            return;
        }

        foreach ($package['update_wpform_ids'] as $update_info) {

            $type = $update_info['type'];
            $name = $update_info['name'];
            $form_name = $update_info['form_name'];

            $new_form_id = $this->get_form_id_by_name($form_name);

            if (!$new_form_id) {

                continue;
            }

            if ($type === 'page') {

                $this->update_form_id_in_page($name, $new_form_id);
            } elseif ($type === 'template') {

                $this->update_form_id_in_template($name, $new_form_id);
            }
        }

        // Update "From Name" and "From Email" settings for every form to be able to send e-mail
        if (!$package || !isset($package['name'])) {

            return;
        }

        $site_url = get_site_url();
        $domain = wp_parse_url($site_url, PHP_URL_HOST);
        $forms = wpforms()->form->get();

        foreach ($forms as $form) {

            $updated = false;
            $form_data = wpforms_decode($form->post_content);

            if (isset($form_data['settings']['notifications']) && is_array($form_data['settings']['notifications'])) {

                // &$notification allows us to make direct changes to $form_data without losing them
                foreach ($form_data['settings']['notifications'] as &$notification) {

                    $notification['sender_name'] = $package['name'];
                    $notification['sender_address'] = "no-reply@$domain";
                    $updated = true;
                }

                if ($updated) {

                    wpforms()->form->update($form->ID, $form_data);
                }
            }
        }
    }

    public function get_form_id_by_name($form_name)
    {

        $forms = wpforms()->form->get();

        foreach ($forms as $form) {

            if ($form->post_title === $form_name) {

                return $form->ID;
            }
        }
        return null;
    }

    public function update_form_id_in_page($page_name, $new_form_id)
    {

        $query = new WP_Query(['post_type' => 'page', 'title' => $page_name, 'posts_per_page' => 1]);

        if ($query->have_posts()) {

            $query->the_post();
            $page_id = get_the_ID();
            $content = get_the_content();

            // Reset post data immediately after we're done using the query results
            wp_reset_postdata();

            $updated_content = $this->replace_form_id_in_content($content, $new_form_id);
            wp_update_post(['ID' => $page_id, 'post_content' => $updated_content]);
        }
    }

    public function replace_form_id_in_content($content, $new_form_id)
    {

        // Replace in Gutenberg blocks
        $content = preg_replace(
            '/"formId":"(\d+)"/',
            '"formId":"' . $new_form_id . '"',
            $content
        );

        // Replace in shortcodes
        $content = preg_replace(
            '/\[wpforms id="(\d+)"/',
            '[wpforms id="' . $new_form_id . '"',
            $content
        );

        return $content;
    }

    public function update_form_id_in_template($template_name, $new_form_id)
    {

        $template_part = get_block_template(get_stylesheet() . '//' . $template_name, 'wp_template_part');

        if ($template_part) {

            $updated_content = $this->replace_form_id_in_content($template_part->content, $new_form_id);
            wp_update_post(['ID' => $template_part->wp_id, 'post_content' => $updated_content]);
        }
    }

    public function update_query_loop_category_ids()
    {

        $pages = get_pages();
        $categories = get_categories(['hide_empty' => false]);
        $category_count = count($categories);

        // For each page, if we find taxQuery (a query loop filtered by category) - then replace category ID with the first category, then on second taxQuery use second category, and so on
        foreach ($pages as $page) {

            $content = $page->post_content;
            $category_index = 0;

            // If there are more taxQuery occurrences on the page than Categories available, then cycle back and start re-using categories from the beginning
            $updated_content = preg_replace_callback(
                '/"taxQuery":\s*{\s*"category":\s*\[(\d+)\]\s*}/',
                function ($matches) use (&$categories, $category_count, &$category_index) {
                    $new_category_id = $categories[$category_index]->term_id;
                    $category_index = ($category_index + 1) % $category_count;
                    return '"taxQuery":{"category":[' . $new_category_id . ']}';
                },
                $content
            );

            if ($content !== $updated_content) {

                wp_update_post([
                    'ID' => $page->ID,
                    'post_content' => $updated_content,
                ]);
            }
        }
    }
}
