<?php

if (!defined('ABSPATH')) {
    exit;
}

if (!function_exists('wp_packages_disable_image_sizes')) {
    /**
     * Disable image sizes
     * 
     * @param string $sizes   List of image sizes.
     *
     * @return array
     */
    function wp_packages_disable_image_sizes($sizes)
    {

        unset($sizes['large']);
        unset($sizes['medium_large']);
        unset($sizes['1536x1536']);
        unset($sizes['2048x2048']);

        return $sizes;
    }
}
add_filter('intermediate_image_sizes_advanced', 'wp_packages_disable_image_sizes');

if (!function_exists('wp_packages_delete_directory')) {
    /**
     * Delete plugin directory with files
     * 
     * @param string $dir   directory.
     *
     * @return boolean
     */
    function wp_packages_delete_directory($dir)
    {
        // Ensure WordPress filesystem functions are available
        require_once(ABSPATH . 'wp-admin/includes/file.php');

        // Initialize the WordPress Filesystem
        if (!function_exists('WP_Filesystem')) {

            require_once(ABSPATH . 'wp-admin/includes/file.php');
        }

        WP_Filesystem();
        global $wp_filesystem;

        if (substr($dir, -1) === '.') {

            return false;
        }

        // Check if the directory exists
        if (!$wp_filesystem->exists($dir)) {

            return false;
        }

        // If it's a single file, delete it normally
        if (!$wp_filesystem->is_dir($dir)) {

            return $wp_filesystem->delete($dir);
        }

        // If it's a folder, recursively delete it and its contents
        return $wp_filesystem->delete($dir, true);
    }
}

if (!function_exists('wpPackagesCleanTerms')) {
    /**
     * Cleanup terms with related data
     *
     * @return void
     */
    function wpPackagesCleanTerms()
    {

        global $wpdb;

        $taxonomies = get_taxonomies();

        foreach ($taxonomies as $taxonomy) {

            $terms = get_terms(array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
            ));

            foreach ($terms as $term) {
                wp_delete_term($term->term_id, $taxonomy);
            }
        }

        $wpdb->query('START TRANSACTION');

        try {
            // Remove all term relationships
            $wpdb->query("TRUNCATE {$wpdb->term_relationships}");

            // Remove all term taxonomies
            $wpdb->query("TRUNCATE {$wpdb->term_taxonomy}");

            // Remove all terms
            $wpdb->query("TRUNCATE {$wpdb->terms}");

            // Reset auto-increment values
            $wpdb->query("ALTER TABLE {$wpdb->terms} AUTO_INCREMENT = 1");
            $wpdb->query("ALTER TABLE {$wpdb->term_taxonomy} AUTO_INCREMENT = 1");

            // Commit the transaction
            $wpdb->query('COMMIT');

            // Optionally, you can recreate the default "Uncategorized" category
            wp_insert_term('Uncategorized', 'category', array('slug' => 'uncategorized'));
        } catch (Exception $e) {

            // If there's an error, roll back the transaction
            $wpdb->query('ROLLBACK');
        }

        // Clear the cache
        wp_cache_flush();
    }
}

if (!function_exists('wpPackagesCleanAttachmentsWithFiles')) {
    /**
     * Delete attachments with files
     *
     * @return void
     */
    function wpPackagesCleanAttachmentsWithFiles()
    {

        $attachments = get_posts([
            'post_type' => 'attachment',
            'post_status' => 'inherit',
            'numberposts' => -1
        ]);

        foreach ($attachments as $attachment) {

            wp_delete_attachment($attachment->ID, true);
        }
    }
}

if (!function_exists('wpPackagesCleanPosts')) {
    /**
     * Cleanup posts table and related data
     *
     * @return void
     */
    function wpPackagesCleanPosts()
    {

        // Delete all other
        $posts = get_posts([
            'post_type' => [
                'wp_navigation',
                'portfolio-works',
                'wp_global_styles',
                'wp_template_part',
                'wpforms',
                'post',
                'page',
            ],
            'post_status' => [
                'publish',
                'pending',
                'draft',
                'auto-draft',
                'future',
                'private'
            ],
            'numberposts' => -1
        ]);

        foreach ($posts as $post) {

            wp_delete_post($post->ID, true);
        }

        // Delete again
        $posts = get_posts(['post_type' => 'any', 'post_status' => array('publish', 'pending', 'draft', 'auto-draft', 'future', 'private'), 'numberposts' => -1]);

        foreach ($posts as $post) {

            wp_delete_post($post->ID, true);
        }
    }
}

if (!function_exists('wpPackagesCleanComments')) {
    /**
     * Cleanup comments table and related data
     *
     * @return void
     */
    function wpPackagesCleanComments()
    {

        $comments = get_comments(array('status' => 'any', 'number' => -1));

        foreach ($comments as $comment) {

            wp_delete_comment($comment->comment_ID, true);
        }
    }
}

if (!function_exists('wpPackDeleteDirectory')) {
    /**
     * Delete any directory with files
     * 
     * @param string $dir   directory.
     *
     * @return void
     */
    function wpPackDeleteDirectory($dir)
    {

        if (!file_exists($dir)) {

            return true;
        }

        if (!is_dir($dir)) {

            return wp_delete_file($dir);
        }

        foreach (scandir($dir) as $item) {

            if ($item == '.' || $item == '..') {

                continue;
            }

            if (!wpPackDeleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {

                return false;
            }
        }

        return rmdir($dir);
    }
}

if (!function_exists('wpPackagesSetPermalinkStructure')) {
    /**
     * Set permalink structure
     * 
     * @param string $structure   permalink structure.
     *
     * @return void
     */
    function wpPackagesSetPermalinkStructure($structure)
    {

        if (!$structure) {

            $structure = '/%postname%/';
        }

        // Set permalinks
        update_option('permalink_structure', $structure);

        flush_rewrite_rules();
    }
}

if (!function_exists('wpPackagesDeleteExactOptions')) {
    /**
     * Delete exact options like site_logo etc.
     * 
     * @param array $options   list of options
     *
     * @return boolean
     */
    function wpPackagesDeleteExactOptions($options = [])
    {

        if (empty($options)) return;

        global $wpdb;

        $placeholders = implode(',', array_fill(0, count($options), '%s'));

        $sql = $wpdb->prepare(
            "DELETE FROM {$wpdb->options} WHERE option_name IN ($placeholders)",
            $options
        );

        return $wpdb->query($sql);
    }
}

if (!function_exists('wpPackagesDeleteOptionsByPattern')) {
    /**
     * Delete options using patterns like: woocommerce_% or %_transient_%.
     * 
     * @param array $patterns   list of options
     *
     * @return void
     */
    function wpPackagesDeleteOptionsByPattern($patterns = [])
    {

        if (!is_array($patterns)) return;

        global $wpdb;

        foreach ($patterns as $pattern) {

            $sql = $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $pattern
            );

            $wpdb->query($sql);
        }
    }
}

if (!function_exists('wpPackagesSanitizePostArray')) {
    /**
     * Sanitize $_POST['array']
     *
     * @return array
     */
    function wpPackagesSanitizePostArray($array)
    {
        if (!is_array($array)) {
            return array();
        }

        return array_map(function ($package) {
            if (is_array($package)) {

                return wpPackagesSanitizePostArray($package);
            } else {

                return sanitize_text_field(wp_unslash($package));
            }
        }, $array);
    }
}
