<?php
/*
 * @wp-packages
 * Plugin Name:       WP Packages
 * Plugin URI:        https://tyler.com/packages/
 * Description:       This plugin will provide an amazing starting point for your new website.
 * Author:            tyler.com
 * Author URI:        https://tyler.com/
 * Version:           1.2
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
	exit;
}

/**
 * Define plugin version.
 */
if (!defined('WP_PACKAGES_PLUGIN_VERSION')) {

	define('WP_PACKAGES_PLUGIN_VERSION', '1.2');
}

/**
 * Define version to use it with JS and CSS files.
 */
if (!defined('WP_PACKAGES_VERSION')) {

	define('WP_PACKAGES_VERSION', '3.4.0');
}

// Debug
if (!function_exists('wpPackagesDebug')) {
	/**
	 * Debug anything. The result will be collected 
	 * in \wp-content\plugins\wpp-generator-v2/mx-debug/mx-debug.txt file
	 * 
	 * @param string $content   List of parameters (coma separated).
	 *
	 * @return void
	 */
	function wpPackagesDebug(...$content)
	{

		$content = wpPackagesContentToString(...$content);

		$dir = plugin_dir_path(__FILE__) . 'mx-debug';

		$file = $dir . '/mx-debug.txt';

		$dateLine = '>>>' . date('Y/m/d H:i:s', time()) . ':' . "\n";

		$current = "{$dateLine}{$content}\n_____________________________________\n";

		if (!file_exists($dir)) {

			mkdir($dir, 0777, true);
		} else {

			$current .= file_get_contents($file) . "\n";
		}

		file_put_contents($file, $current);
	}
}

if (!function_exists('wpPackagesContentToString')) {
	/**
	 * This function is a part of the
	 * wpPackagesDebug function.
	 * 
	 * @param string $content   List of parameters (coma separated).
	 *
	 * @return string
	 */
	function wpPackagesContentToString(...$content)
	{

		ob_start();

		var_dump(...$content);

		return ob_get_clean();
	}
}

require_once plugin_dir_path(__FILE__) . 'includes/functions.php';

// Fix for WordPress 6.7.0+ early text domain loading issue
add_action('plugins_loaded', function() {
    // Suppress the specific notice for portfolio-pack domain during package installation
    add_filter('doing_it_wrong_trigger_error', function($trigger, $function, $message) {
        if ($function === '_load_textdomain_just_in_time') {
            // Check if this is related to portfolio-pack domain
            if (strpos($message, 'portfolio-pack') !== false) {
                return false; // Don't trigger the error for portfolio-pack theme
            }
        }
        return $trigger;
    }, 10, 3);
}, 1);

// Class used to suppress output during WP plugin installations
require_once(ABSPATH . 'wp-admin/includes/class-wp-upgrader.php');

class Quiet_Skin extends \WP_Upgrader_Skin
{

	public function feedback($string, ...$args) {}
	public function header() {}
	public function footer() {}
}

// WP_CRON
require_once plugin_dir_path(__FILE__) . 'includes/wp-cron-manager.php';

// Endpoints
require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step1.php';
new HandleStepOne;

require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step2.php';
new HandleStepTwo;

require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step3.php';
new HandleStepThree;

require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step4.php';
new HandleStepFour;

require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step5.php';
new HandleStepFive;

require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step6.php';
new HandleStepSix;

require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step7.php';
new HandleStepSeven;

require_once plugin_dir_path(__FILE__) . 'includes/endpoints/step8.php';
new HandleStepEight;

class WP_Package_Importer
{

	private $packages;

	public function __construct()
	{

		// Load text domain properly at init
		add_action('init', array($this, 'load_textdomain'));

		// Include package-config.php and enqueue CSS/JS scripts
		$this->packages = require plugin_dir_path(__FILE__) . 'package-config.php';
		add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

		// Register See Packages link on Plugins page & register the Admin menu
		add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_action_links'));
		add_action('admin_menu', array($this, 'create_admin_menu'));

		// Ensure themes load text domains properly during package installation
		add_action('after_switch_theme', array($this, 'ensure_theme_textdomain_loading'));
	}

	/**
	 * Load plugin text domain
	 */
	public function load_textdomain()
	{
		load_plugin_textdomain('wp-packages', false, dirname(plugin_basename(__FILE__)) . '/languages/');
	}

	/**
	 * Ensure theme text domain loads properly after theme switch
	 */
	public function ensure_theme_textdomain_loading()
	{
		// Add a hook to delay any early text domain loading attempts
		add_action('init', array($this, 'load_any_pending_textdomains'), 30);
	}

	/**
	 * Load any pending text domains that might have been triggered early
	 */
	public function load_any_pending_textdomains()
	{
		// This ensures that any text domains that were attempted to load early
		// get properly loaded at the init hook or later
		$theme = wp_get_theme();
		if ($theme && $theme->get_stylesheet()) {
			// Safely load theme text domain if it exists
			$theme_textdomain = $theme->get('TextDomain');
			if ($theme_textdomain) {
				load_theme_textdomain($theme_textdomain, get_template_directory() . '/languages');
			}
		}
	}

	public function enqueue_admin_scripts($hook)
	{

		// Only load on the plugin's admin page
		if ('toplevel_page_wp-packages' !== $hook) {

			return;
		}

		// Define a version number
		$version = WP_PACKAGES_VERSION;

		// Enqueue plugin CSS file
		wp_enqueue_style('wp-packages-admin-style',  plugins_url('assets/css/admin-style.css', __FILE__), array('wp-jquery-ui-dialog'), $version);

		// Enqueue the admin-setup.js file
		wp_enqueue_script('wp-packages-admin-setup', plugins_url('assets/js/admin-setup.js', __FILE__), array('jquery', 'jquery-ui-dialog'), $version, true);

		// todo I will need to clean this up, no need to send all this
		// Localize the script with new data
		wp_localize_script(
			'wp-packages-admin-setup',
			'wpPackageImporter',
			array(
				'plugin_version' => WP_PACKAGES_PLUGIN_VERSION,
				'sseUrl' => admin_url('admin-post.php?action=wp_package_sse'),
				'nonce' => wp_create_nonce('install_wp_package'),
				'ajax_url' => admin_url('admin-ajax.php'),
				'siteUrl' => home_url(),
				'packages' => $this->packages,
				'initMessage' => 'Preparing to install...',
				'wpVersion' => get_bloginfo('version'),
				'phpVersion' => phpversion(),
				'root' => esc_url_raw(rest_url())
			)
		);

		// Enqueue the install-handler.js file
		wp_enqueue_script('wp-packages-install-handler', plugins_url('assets/js/install-handler.js', __FILE__), array('jquery', 'wp-packages-admin-setup'), $version, true);

		// Add a random query string to force fresh load todo remove
		wp_add_inline_script('wp-packages-admin-setup', 'var forceReload = "' . uniqid() . '";', 'before');
		wp_add_inline_script('wp-packages-install-handler', 'var forceReload = "' . uniqid() . '";', 'before');
	}

	public function add_action_links($links)
	{

		$custom_links = array(
			'<a href="' . admin_url('admin.php?page=wp-packages') . '">See Packages</a>'
		);

		return array_merge($custom_links, $links);
	}

	public function create_admin_menu()
	{

		add_menu_page(
			'WP Package Importer',
			'WP Packages',
			'manage_options',
			'wp-packages',
			array($this, 'admin_page'),
			'dashicons-wordpress-alt',
			150
		);
	}

	public function admin_page()
	{

?>
		<div class="blog-package-wrap">
			<h1>WordPress Packages</h1>
			<div class="flex-container">
				<?php foreach ($this->packages as $package_id => $package) : ?>
					<div>
						<img src="<?php echo esc_url(plugins_url('assets/images/' . $package['image'], __FILE__)); ?>" alt="<?php echo esc_attr($package['name']); ?> Image">
						<h2><?php echo esc_html($package['name']); ?></h2>
						<p><?php echo esc_html($package['description']); ?></p>
						<div class="button-container">
							<?php if ($package['isLive']) : ?>
								<button class="button button-primary package-input install-wp-package" data-package-id="<?php echo esc_attr($package_id); ?>">Install <?php echo esc_html($package['name']); ?></button>
							<?php endif; ?>
							<?php if ($package['isLive'] && !empty($package['tutorial_url'])) : ?>
								<a href="<?php echo esc_url($package['tutorial_url']); ?>" target="_blank" class="button button-secondary watch-video">
									Watch Tutorial
								</a>
							<?php else : ?>
								<a href="#" class="button button-secondary watch-video disabled" onclick="return false;">
									Coming soon...
								</a>
							<?php endif; ?>
						</div>
					</div>
				<?php endforeach; ?>
			</div>
		</div>

		<div id="confirm-dialog" title="Confirm Installation" style="display: none;">
			<p>You've chosen the <b id="selected-package-name"></b>.</p>
			<p>This package is only meant to be installed on a fresh website. Please confirm that you'd like to proceed.</p>
			<p id="status-message"><b>This will delete and replace your entire website!</b></p>
		</div>
<?php

	}
}

new WP_Package_Importer();
